function locus = symlocd(b,a,r1,r2,dsmax,scale)
% SYMLOCD Plots symmetric root locus for discrete system
%
%	locus = symlocd(b,a,r1,r2,dsmax,scale)
%	locus = symlocd(b,a,r1,r2,dsmax)
%
%	Plots the stable part of the root locus corresponding to
%	a(z)*a(1/z) + r*b(z)*b(1/z) for r1 < r < r2. Also the roots of
%       a (r = Inf) and the roots of b (r = 0) are plotted.
%
%       The increments in r are calculated such that abs(ds) < dsmax.
%       The r-values and the roots are stored in locus, with the r-values
%       in the first column.
%
%	scale (optional) takes the form [xmax xmin ymin ymax]. If scale is 
%	omitted an attempt is done to find a plot scale that gives an aspect
%	ratio as close to 1 as possible.
%
%       Use PZGRID(1) to plot grid and unit circle.

% Kjell Gustafsson, original file due to Ulf Holmberg
% LastEditDate : Thu May 30 09:17:11 1991
% Copyright (c) 1990 by Kjell Gustafsson and Department of Automatic Control,
% Lund Institute of Technology, Lund, SWEDEN

aspec = conv(a,a(length(a):-1:1));
bspec = conv(b,b(length(b):-1:1));

reldeg = length(a) - length(b);
if reldeg>=0,
  b = [0*(1:reldeg) b];
else
  a = [0*(1:-reldeg) a];
end

ainv = a(length(a):-1:1);
binv = b(length(b):-1:1);
asym = conv(b,binv);
bsym = conv(a,ainv);
while asym(length(asym))==0 & bsym(length(bsym))==0
  asym = asym(1:length(asym)-1);
  bsym = bsym(1:length(bsym)-1);
end

loc = rootloc(bsym,asym,r1,r2,dsmax,1);

% sort out stable roots

[n,m] = size(loc);
locus = [];
for k = 1:n,
  [imvec,imindex] = sort(imag(loc(k,2:m)));
  loc(k,2:m) = loc(k,imindex+1);
  index1 = find(abs(loc(k,2:m))==1);
  index = find(abs(loc(k,2:m))<1);
  locus = [locus; loc(k,1) loc(k,index+1) loc(k,index1(1:2:length(index1))+1)];
end

rl = locus(:,2:size(locus)*[0;1]);

po = roots(aspec);
[tmp,ind] = sort(abs(po));
po = po(ind(1:length(ind)/2));
ze = roots(bspec);
[tmp,ind] = sort(abs(ze));
ze = ze(ind(1:length(ind)/2));

if reldeg>=0,
  ze = [zeros(reldeg,1); ze];
else
  po = [zeros(-reldeg,1); po];
end

if nargin<6
  Re_max = max([max(real(rl)) real(po') real(ze')]);
  Re_min = min([min(real(rl)) real(po') real(ze')]);
  Im_max = max([max(imag(rl)) imag(po') imag(ze')]);
  dRe = Re_max - Re_min;
  dIm = 2*Im_max;
  mRe = (Re_min + Re_max)/2;
  d = max(dRe,dIm);
  if d==0
    w = [-abs(Re_max) abs(Re_max)]*1.2;
  else
    w = [-d/2 d/2]*1.2;
  end
  scale = [w+mRe w];
end

axis('square');
axis(scale);
plot(0,0,'i');
hold on;
plot(real(rl),imag(rl),'.w');
mark(ze,4,[],[],1);
mark(po,2,[],[],1);
xlabel('Re');
ylabel('Im');
hold off;
axis('normal');
