function tryu=yusimd(bp,ap,r,s,t,bff,aff,h,tmax)
% YUSIM Simulation of discrete time SISO system.
%
%       TRYU=YUSIMD(B,A,R,S,T,TRLDN)
%       TRYU=YUSIMD(B,A,R,S,T,H,TMAX)
%       TRYU=YUSIMD(B,A,R,S,T,BFF,AFF,TRLDN)
%       TRYU=YUSIMD(B,A,R,S,T,BFF,AFF,H,TMAX)
%
%       The strictly proper system
%
%       y(z) = B(z)/A(z)*(u(z)+l(z)) + d(z)
%
%       is controlled using
%
%       R(z)*u(z) = T(z)*BFF(z)/AFF(z)*ref(z)
%                    - S(z)*(y(z)+n(z))
%
%       The feedforward filter BFF/AFF is optional.
%
%       If B has n rows and A has 1 row then n systems B(i,:)/A are
%       simulated with the same controller.  If B has 1 row and A has
%       n rows then n systems B/A(i,:) are simulated.  If B and A both
%       have n rows then n systems B(i,:)/A(i,:) are simulated.  This 
%       feature is nice for simulation of systems with parametric
%       uncertainty.
%
%       System output and control signal are calculated and stored in
%       the matrix TRYU=[TIME REF Y1 .. Yn U1 .. Un]. This time response
%       is displayed using YUPL and YUSH.
%
%       Argument TRLDN defines simulation time and external signals.
%       It may have different formats:
%         Scalar TMAX:  Simulation time defined by vector 0:TMAX where
%                       TMAX is a positive integer. Sampling interval is 1.
%                       Reference signal r=1. An input disturbance l=-1
%                       affects the system from TMAX/3 and an output
%                       disturbance d=-1 affects the system from TMAX*2/3
%                       No noise is present.
%         Vector TIME:  Simulation time is defined. Signals as above.
%                       Time increment defines the sampling interval.
%         Matrix TRLDN: A matrix with 2 - 5 columns.
%                       First column defines simulation time. Second
%                       column defines the reference signal r at time-
%                       instants of first column. If present column 
%                       three to five define input disturbance l, output
%                       disturbance d and measurement noise n respectively.
%                       Omitted column implies that the corresponding 
%                       signal is zero.
%                       The matrix TRLDN may be generated by YUSIGNALS.
%
%       Two scalars H and TMAX may instead be given to form TRLDN.  H is
%       the sampling interval.  Simulation time id defined by vector
%       0:H:TMAX.  Signals as for scalar TMAX in the definition of TRLDN.

% Michael Lundh     LastEditDate : Wed Mar  7 16:39:54 1990
% Copyright (c) 1990 by Michael Lundh and Department of Automatic Control,
% Lund Institute of Technology, Lund, SWEDEN

if nargin==6,      trldn=bff; bff=1; aff=1;
                   if length(trldn)==1
                     trldn=(0:trldn)';
                   end
elseif nargin==7,  trldn=(0:bff:aff)'; bff=1; aff=1;
elseif nargin==8,  trldn=h;
                   if length(trldn)==1
                     trldn=(0:trldn)';
                   end
elseif nargin==9,  trldn=(0:h:tmax)';
else               error('False number of imputs')
end;

%---- assign trldn ----
[m,n]=size(trldn);
if (n==1) | (m==1)            % only time specified
  trldn = trldn(:);
  m     = max(m,n);
  trldn = [trldn ones(m,1) zeros(m,3)];
  mi    = floor(m/3)+1;
  trldn(mi:m,3) = -ones(m-mi+1,1);
  mi    = floor(2*m/3)+1;
  trldn(mi:m,4) = -ones(m-mi+1,1);
else
  trldn=[trldn zeros(m,5-n)];
end

%---- RST-controller ----
[fb,hb,gb1,kb1] = tf2ss(t,r);
[fb,hb,gb2,kb2] = tf2ss(s,r);
fb  = fb' ;
hb  = hb' ;
gb1 = gb1';
gb2 = gb2';


%---- FF-filter ----
[ff,gf,hf,kf] = tf2ss(bff,aff);
mff=size(ff)*[1;0];

nbpol = size(bp)*[1;0];
napol = size(ap)*[1;0];
if napol~=nbpol
  if napol==1,     ap = kron(ap,ones(nbpol,1));
  elseif nbpol==1, bp = kron(bp,ones(napol,1));
  else error('More than 1 row in A and B and not same number of rows')
  end
end
nbpol = size(bp)*[1;0];

disp(sprintf('Simulation of %g system(s)',nbpol))
yy=[];uu=[];
for i=1:nbpol
  %---- Open system ----
  api = ap(i,:);
  bpi = bp(i,:);
  while abs(bpi(1))<10000*eps, bpi(1)=[]; end
  if length(api)<=length(bpi)
    error('B/A is not strictly proper')
  end

  [a,b,c,d]=tf2ss(bpi,api);

  if length(kb2) > 0,
    a11 = a-b*kb2*c;
  else
    a11 = [];
  end;
  if length(hb) > 0,
    a12 = b*hb;
  else
    a12 = [];
  end
  if length(gb2) > 0,
    a21 = -gb2*c;
  else
    a21 = [];
  end;
  aa = [a11 a12 ; a21 fb];
  [maa,naa]=size(aa);
  if length(hf)>0,
    aa1 = b*kb1*hf;
    aa2 = gb1*hf;
  else
    aa1 = [];
    aa2 = [];
  end;

  [maa,naa]=size(aa);

  aaa=[aa [aa1 ; aa2] ; zeros(mff,naa) ff];
  bbb=[b*kb1*kf b -b*kb2 -b*kb2  ; gb1*kf 0*gb2 -gb2 -gb2 ; gf 0*[gf gf gf]];
  ccc=[ c 0*hb 0*hf ; -kb2*c hb kb1*hf];
  ddd=[0 0 1 0 ; kb1*kf 0 -kb2 -kb2];

  yu = dlsim(aaa,bbb,ccc,ddd,trldn(:,2:5));
  yy = [yy yu(:,1)];
  uu = [uu yu(:,2)];
  disp(sprintf(' %g done',i))
end

tryu=[trldn(:,1:2) yy uu];
