function [fry,frref] = frdpid(k,ti,td,n,b,tsamp,lgw1,lgw2,np)
% FRDPID Computes the frequency response of a discrete time PID controller
%
%       [fry,frref] = frdpid(k,ti,td,n,b,tsamp,lgw1,lgw2,np)
%       [fry,frref] = frdpid(k,ti,td,n,b,tsamp,wvec)
%
%       There are two transfer functions associated with a PID controller:
%
%         Gr(s) = k*(b + 1/(ti*s))                    (reference value, frref)
%         Gy(s) = k*(1 + 1/(ti*s) + s*td/(s*td/n+1))) (measured value, fry)
%
%	with the control signal formed as u = Gr r - Gy y. A standard
%	discretization is to use forward Euler for the I-part and backward
% 	Euler for the derivative, i.e. 1/(ti*s) -> tsamp/(ti*(z-1)) and
%	s*td/(s*td/n+1) -> n*gamma*(z-1)/(z-gamma) with gamma = td/(td+n*tsamp)
%	
%	Using these discretizations, the frequency response for the two
%	transfer functions are calculated for the values in wvec [rad/s] or np
%       logarithmically spaced frequency points [rad/s] between 10^lgw1 and
%       10^lgw2. The argument np is optional with default value 50. If lgw2 is
%       supplied as an empty matrix or omitted it is taken as half the
%       sampling frequency (pi/tsamp). The output frref may be omitted.
%
%	k, ti, td, n, and/or b may contain several rows. The frequency response
%       for each set k(i), ti(i), td(i), n(i) and b(i) is calculated. If any of
%       the parameters is given as a single value while the others contain
%       several rows, then this single value is used for all the rows in the 
%	other parameters.
%
%	The output fr takes the form [ w  H1(iw) H2(iw) ...] where Hi
%       corresponds to row i of k, ti, td, n and b.
%
%       The frequency response of a PI controller can be calculated
%       by setting td to []. The filter on the D-part is removed by setting
%	n to [] and a PD controller is calculated by setting ti to [].

% Lars Rundqwist     
% LastEditDate : Wed Jun 13 11:51:42 1990
% Copyright (c) 1990 by Lars Rundqwist and Department of Automatic Control,
% Lund Institute of Technology, Lund, SWEDEN
% Modification of FRCPID (using FRD) by Kjell Gustafsson

i = sqrt(-1);

if nargin==9
  if size(lgw2)~=[0 0]
    w = logspace(lgw1,lgw2,np)';
  else
    w = logspace(lgw1,log10(pi/tsamp),np)';
  end
elseif nargin==8
  if size(lgw2)~=[0 0]
    w = logspace(lgw1,lgw2)';
  else
    w = logspace(lgw1,log10(pi/tsamp))';
  end
elseif length(lgw1)==1
  w = logspace(lgw1,log10(pi/tsamp))';
else
  w = lgw1(:);
end

nk = length(k);
nti = length(ti);
ntd = length(td);
nn = length(n);
nb = length(b);

maxn = max([nk nti ntd nn nb]);

if [nk nti ntd nn nb]==maxn | [nk nti ntd nn nb]==1 | [nk nti ntd nn nb]==0
  if nk==1, k = kron(k,ones(maxn,1)); end
  if nti==1, ti = kron(ti,ones(maxn,1)); end
  if ntd==1, td = kron(td,ones(maxn,1)); end 
  if nn==1, n = kron(n,ones(maxn,1)); end 
  if nb==1, b = kron(b,ones(maxn,1)); end 
else
  error('Inconsistent number of rows in k, ti, td, n, and b')
end

k = k(:);
ti = ti(:);
td = td(:);
n = n(:);
b = b(:);

gamma=td./(n*tsamp+td);
onesm=ones(maxn,1);

if [nti ntd]==0
  % P
  fry = [w gval(k,1,0,exp(i*w*tsamp))];
  frref = [w gval(k.*b,1,0,exp(i*w*tsamp))];
elseif [nti nn]==0
  % PD without filter
  fry = [w gval([k.*td/tsamp k-k.*td/tsamp],1,0,exp(i*w*tsamp))];
  frref = [w gval(k.*b,1,0,exp(i*w*tsamp))];
elseif nti==0
  % PD with filter
  fry = [w gval([k.*(n.*gamma+onesm) -k.*(n.*gamma+gamma)],[onesm -gamma], ...
          0,exp(i*w*tsamp))];
  frref = [w gval(k.*b,1,0,exp(i*w*tsamp))];
elseif ntd==0
  % PI
  fry = [w gval([k  k.*(tsamp*onesm./ti-onesm)],[1 -1],0,exp(i*w*tsamp))];
  frref = [w gval([k.*b  k.*(tsamp*onesm./ti-b)],[1 -1],0,exp(i*w*tsamp))];
elseif nn==0
  % PID without filter
  fry = [w gval(...
    [k.*td/tsamp k.*(onesm-2*td/tsamp) k.*(tsamp*onesm./ti-onesm+td/tsamp)],...
    [1 -1],0,exp(i*w*tsamp))];
  frref = [w gval([k.*b  k.*(tsamp*onesm./ti-b)],[1 -1],0,exp(i*w*tsamp))];
else
  % PID
  fry = ...
    [w gval(...
    [k.*(n.*gamma+onesm) k.*(tsamp*onesm./ti-onesm-gamma-2*n.*gamma)..
       k.*(gamma-tsamp*gamma./ti+n.*gamma)],..
    [onesm -(onesm+gamma) gamma],0,exp(i*w*tsamp))];
  frref = [w gval([k.*b  k.*(tsamp*onesm./ti-b)],[1 -1],0,exp(i*w*tsamp))];
end
