function [fry,frref] = frcpid(k,ti,td,n,b,lgw1,lgw2,np)
% FRCPID Computes the frequency response of a continuous-time PID controller
%
%       [fry,frref] = frcpid(k,ti,td,n,b,lgw1,lgw2,np)
%       [fry,frref] = frcpid(k,ti,td,n,b,wvec)
%
%       There are two transfer functions associated with a PID controller:
%
%         Gr(s) = k*(b + 1/(ti*s))                    (reference value, frref)
%         Gy(s) = k*(1 + 1/(ti*s) + s*td/(s*td/n+1))) (measured value, fry)
%
%	with the control signal formed as u = Gr r - Gy y.
%       The frequency response for these transfer functions are calculated
%	for the values in wvec [rad/s] or np logarithmically spaced frequency 
%       points [rad/s] between 10^lgw1 and 10^lgw2. The argument np is optional
%       with default value 50. The output frref may be omitted.
%
%	k, ti, td, n, and/or b may contain several rows. The frequency response
%       for each set k(i), ti(i), td(i), n(i) and b(i) is calculated. If any of
%       the parameters is given as a single value while the others contain
%       several rows, then this single value is used for all the rows in the 
%	other parameters.
%
%	The output fr takes the form [ w  G1(iw) G2(iw) ...] where Gi
%       corresponds to row i of k, ti, td, n and b.
%
%       The frequency response of a PI controller can be calculated
%       by setting td to []. The filter on the D-part is removed by setting
%	n to [] and a PD controller is calculated by setting ti to [].

% Kjell Gustafsson     
% LastEditDate : Wed Jun 13 08:38:02 1990
% Copyright (c) 1990 by Kjell Gustafsson and Department of Automatic Control,
% Lund Institute of Technology, Lund, SWEDEN

i = sqrt(-1);

if nargin==8
  w = logspace(lgw1,lgw2,np)';
elseif nargin==7
  w = logspace(lgw1,lgw2)';
else
  w = lgw1(:);
end

nk = length(k);
nti = length(ti);
ntd = length(td);
nn = length(n);
nb = length(b);

maxn = max([nk nti ntd nn nb]);

if [nk nti ntd nn nb]==maxn | [nk nti ntd nn nb]==1 | [nk nti ntd nn nb]==0
  if nk==1, k = kron(k,ones(maxn,1)); end
  if nti==1, ti = kron(ti,ones(maxn,1)); end
  if ntd==1, td = kron(td,ones(maxn,1)); end 
  if nn==1, n = kron(n,ones(maxn,1)); end 
  if nb==1, b = kron(b,ones(maxn,1)); end 
else
  error('Inconsistent number of rows in k, ti, td, n, and b')
end

k = k(:);
ti = ti(:);
td = td(:);
n = n(:);
b = b(:);

if [nti ntd]==0
  % P
  fry = [w gval(k,1,0,i*w)];
  frref = [w gval(k.*b,1,0,i*w)];
elseif [nti nn]==0
  % PD without filter
  fry = [w gval([k.*td k],1,0,i*w)];
  frref = [w gval(k.*b,1,0,i*w)];
elseif nti==0
  % PD with filter
  fry = [w gval([k.*(n+ones(maxn,1)) k.*n./td],[ones(maxn,1) n./td],0,i*w)];
  frref = [w gval(k.*b,1,0,i*w)];
elseif ntd==0
  % PI
  fry = [w gval([k k./ti],[1 0],0,i*w)];
  frref = [w gval([k.*b k./ti],[1 0],0,i*w)];
elseif nn==0
  % PID without filter
  fry = [w gval([k.*td k k./ti],[1 0],0,i*w)];
  frref = [w gval([k.*b k./ti],[1 0],0,i*w)];
else
  % PID
  fry = ...
    [w gval([k.*(n+ones(maxn,1)) k.*(n./td+ones(maxn,1)./ti) k.*n./(ti.*td)],..
    [ones(maxn,1) n./td zeros(maxn,1)],0,i*w)];
  frref = [w gval([k.*b k./ti],[1 0],0,i*w)];
end

